<?php
// =============================================
// StudyTrack LK — AI Feedback Engine
// Hybrid: PHP Rules + Claude API backup
// =============================================

// ---- CLAUDE API KEY ---
// Get your key from: https://console.anthropic.com
define('CLAUDE_API_KEY', 'your_claude_api_key_here');
define('CLAUDE_MODEL',   'claude-opus-4-6');

// =============================================
// MAIN: Generate full AI report for a student
// =============================================
function generateAIReport($conn, $uid) {
    $data = collectStudentData($conn, $uid);

    // Step 1: PHP rules-based analysis (instant, no API needed)
    $rules = runRulesEngine($data);

    // Step 2: Claude API deep analysis (if API key is set)
    $aiInsight = '';
    if (defined('CLAUDE_API_KEY') && CLAUDE_API_KEY !== 'your_claude_api_key_here') {
        $aiInsight = callClaudeAPI($data, $rules);
    }

    return [
        'data'      => $data,
        'rules'     => $rules,
        'ai_insight'=> $aiInsight,
        'generated' => date('Y-m-d H:i:s'),
    ];
}

// =============================================
// Collect all student data from DB
// =============================================
function collectStudentData($conn, $uid) {
    $uid = (int)$uid;

    // Basic info
    $user = mysqli_fetch_assoc(mysqli_query($conn, "SELECT * FROM users WHERE id=$uid"));

    // Total & weekly hours
    $totalH = (float)mysqli_fetch_assoc(mysqli_query($conn,
        "SELECT COALESCE(SUM(hours),0) as h FROM study_sessions WHERE user_id=$uid"))['h'];
    $weekH  = (float)mysqli_fetch_assoc(mysqli_query($conn,
        "SELECT COALESCE(SUM(hours),0) as h FROM study_sessions WHERE user_id=$uid
         AND session_date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)"))['h'];

    // Subject-wise hours
    $subHours = [];
    $r = mysqli_query($conn,
        "SELECT subject, SUM(hours) as h FROM study_sessions WHERE user_id=$uid GROUP BY subject ORDER BY h DESC");
    while ($row = mysqli_fetch_assoc($r)) $subHours[$row['subject']] = round((float)$row['h'], 1);

    // Paper performance per subject
    $paperPerf = [];
    $r = mysqli_query($conn,
        "SELECT subject,
            COUNT(*) as cnt,
            AVG((marks_obtained/total_marks)*100) as avg_pct,
            MIN((marks_obtained/total_marks)*100) as min_pct,
            MAX((marks_obtained/total_marks)*100) as max_pct
         FROM past_papers WHERE user_id=$uid GROUP BY subject");
    while ($row = mysqli_fetch_assoc($r)) {
        $paperPerf[$row['subject']] = [
            'count'   => (int)$row['cnt'],
            'avg_pct' => round((float)$row['avg_pct'], 1),
            'min_pct' => round((float)$row['min_pct'], 1),
            'max_pct' => round((float)$row['max_pct'], 1),
            'grade'   => gradeFromPercent((float)$row['avg_pct']),
        ];
    }

    // Paper trend (last 10 papers chronologically)
    $paperTrend = [];
    $r = mysqli_query($conn,
        "SELECT subject, done_date, marks_obtained, total_marks,
            ROUND((marks_obtained/total_marks)*100,1) as pct
         FROM past_papers WHERE user_id=$uid ORDER BY done_date ASC, id ASC LIMIT 10");
    while ($row = mysqli_fetch_assoc($r)) $paperTrend[] = $row;

    // Streak
    $streakRow = mysqli_fetch_assoc(mysqli_query($conn,
        "SELECT current_streak, longest_streak FROM streaks WHERE user_id=$uid"));
    $streak  = $streakRow ? (int)$streakRow['current_streak']  : 0;
    $longest = $streakRow ? (int)$streakRow['longest_streak']  : 0;

    // Goals
    $goals = [];
    $r = mysqli_query($conn, "SELECT * FROM goals WHERE user_id=$uid");
    while ($row = mysqli_fetch_assoc($r)) $goals[$row['subject']] = $row;

    // Study days last 30 days
    $studyDays = (int)mysqli_fetch_assoc(mysqli_query($conn,
        "SELECT COUNT(DISTINCT session_date) as d FROM study_sessions
         WHERE user_id=$uid AND session_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)"))['d'];

    // Average score overall
    $avgScore = (float)(mysqli_fetch_assoc(mysqli_query($conn,
        "SELECT COALESCE(AVG((marks_obtained/total_marks)*100),0) as a FROM past_papers WHERE user_id=$uid"))['a']);

    return compact(
        'user','totalH','weekH','subHours','paperPerf',
        'paperTrend','streak','longest','goals','studyDays','avgScore'
    );
}

// =============================================
// RULES ENGINE — Pure PHP logic
// =============================================
function runRulesEngine($data) {
    $result = [
        'weak_subjects'       => [],
        'strong_subjects'     => [],
        'neglected_subjects'  => [],
        'tips'                => [],
        'grade_predictions'   => [],
        'study_recommendations'=> [],
        'trend_analysis'      => '',
        'overall_status'      => '',
        'motivation'          => '',
        'score'               => 0,   // 0-100 overall health score
    ];

    $pp   = $data['paperPerf'];
    $sh   = $data['subHours'];
    $goals= $data['goals'];

    // ---- Identify weak / strong subjects ----
    foreach ($pp as $subj => $perf) {
        if ($perf['avg_pct'] < 50)      $result['weak_subjects'][]   = ['subject'=>$subj, 'avg'=>$perf['avg_pct'], 'grade'=>$perf['grade']];
        elseif ($perf['avg_pct'] >= 70) $result['strong_subjects'][] = ['subject'=>$subj, 'avg'=>$perf['avg_pct'], 'grade'=>$perf['grade']];
    }
    usort($result['weak_subjects'],   fn($a,$b) => $a['avg'] <=> $b['avg']);
    usort($result['strong_subjects'], fn($a,$b) => $b['avg'] <=> $a['avg']);

    // ---- Neglected subjects (goal set but no hours logged) ----
    foreach ($goals as $subj => $g) {
        $hrs = $sh[$subj] ?? 0;
        if ($hrs < 2 && !isset($pp[$subj])) {
            $result['neglected_subjects'][] = $subj;
        }
    }

    // ---- Grade predictions ----
    foreach ($pp as $subj => $perf) {
        $trend   = getSubjectTrend($data['paperTrend'], $subj);
        $predPct = $perf['avg_pct'];
        if ($trend === 'improving')    $predPct = min(100, $predPct + 5);
        elseif ($trend === 'declining') $predPct = max(0, $predPct - 5);
        $result['grade_predictions'][$subj] = [
            'predicted_grade' => gradeFromPercent($predPct),
            'predicted_pct'   => round($predPct, 1),
            'trend'           => $trend,
            'current_avg'     => $perf['avg_pct'],
        ];
    }

    // ---- Trend analysis (overall) ----
    if (count($data['paperTrend']) >= 3) {
        $scores = array_column($data['paperTrend'], 'pct');
        $n      = count($scores);
        $first3 = array_sum(array_slice($scores, 0, 3)) / 3;
        $last3  = array_sum(array_slice($scores, -3))   / 3;
        if ($last3 > $first3 + 5)      $result['trend_analysis'] = 'improving';
        elseif ($last3 < $first3 - 5)  $result['trend_analysis'] = 'declining';
        else                            $result['trend_analysis'] = 'stable';
    } else {
        $result['trend_analysis'] = 'insufficient_data';
    }

    // ---- Study recommendations ----
    // Prioritize weak subjects
    foreach ($result['weak_subjects'] as $ws) {
        $currHrs  = $sh[$ws['subject']] ?? 0;
        $recHrs   = $ws['avg'] < 35 ? 4 : ($ws['avg'] < 50 ? 3 : 2);
        $result['study_recommendations'][] = [
            'subject'       => $ws['subject'],
            'current_hours' => $currHrs,
            'recommend_hours'=> $recHrs,
            'reason'        => 'Weak subject — needs more attention',
            'priority'      => 'HIGH',
        ];
    }
    // Moderate subjects need maintenance
    foreach ($pp as $subj => $perf) {
        if ($perf['avg_pct'] >= 50 && $perf['avg_pct'] < 70) {
            $result['study_recommendations'][] = [
                'subject'        => $subj,
                'current_hours'  => $sh[$subj] ?? 0,
                'recommend_hours'=> 2,
                'reason'         => 'Average subject — maintain consistency',
                'priority'       => 'MEDIUM',
            ];
        }
    }

    // ---- Subject-specific tips ----
    $subjectTips = [
        'Combined Maths' => ['Practice integration & differentiation daily','Do at least 2 MCQ papers per week','Focus on Series & Probability for quick marks','Work through 2019-2023 papers systematically'],
        'Physics'        => ['Draw diagrams for every problem','Master vectors, mechanics, and electricity','Practice MCQ paper under timed conditions','Re-derive formulas to understand, not memorize'],
        'Chemistry'      => ['Make a periodic table summary card','Organic chemistry needs daily revision','Balance equations step by step','Past papers from 2018+ are highly representative'],
        'Biology'        => ['Draw and label diagrams — examiners reward detail','Memorize life cycles and processes flowcharts','Structured answers need correct technical terms','Do at least 1 essay per week'],
        'ICT'            => ['Practice SQL queries and algorithms','Memorize OSI model layers','Draw network diagrams neatly','Theory + practical both tested equally'],
        'Economics'      => ['Learn all definitions precisely','Use diagrams (AD-AS, supply-demand) in every answer','Memorize key statistics: GDP, inflation rates','Link theory to real Sri Lanka examples'],
        'Accounting'     => ['Practice journal entries daily','Double-check debit=credit every time','Ratio analysis needs formula memorization','Past paper marking schemes are gold'],
    ];
    foreach ($result['weak_subjects'] as $ws) {
        $subj = $ws['subject'];
        $tips = $subjectTips[$subj] ?? ['Practice more past papers','Revise fundamentals','Seek help from teacher on weak topics'];
        $result['tips'][$subj] = $tips;
    }

    // ---- Overall status & score ----
    $healthScore = 50;
    if ($data['weekH'] >= 14)       $healthScore += 10;
    elseif ($data['weekH'] >= 7)    $healthScore += 5;
    else                            $healthScore -= 10;

    if ($data['streak'] >= 7)       $healthScore += 10;
    elseif ($data['streak'] >= 3)   $healthScore += 5;

    if ($data['avgScore'] >= 65)    $healthScore += 15;
    elseif ($data['avgScore'] >= 50)$healthScore += 8;
    else                            $healthScore -= 10;

    if ($data['studyDays'] >= 20)   $healthScore += 10;
    elseif ($data['studyDays'] >= 10)$healthScore += 5;

    if (!empty($result['weak_subjects'])) $healthScore -= count($result['weak_subjects']) * 5;
    $result['score'] = max(0, min(100, $healthScore));

    if ($result['score'] >= 75)     $result['overall_status'] = 'excellent';
    elseif ($result['score'] >= 55) $result['overall_status'] = 'good';
    elseif ($result['score'] >= 35) $result['overall_status'] = 'needs_improvement';
    else                            $result['overall_status'] = 'critical';

    // ---- Motivational message ----
    $messages = [
        'excellent'        => ["Outstanding work! 🌟 You're on track for your dream university.", "Keep this momentum — you're performing at a top level! 🔥", "Excellent progress. Your dedication is showing in your results! 💪"],
        'good'             => ["Good progress! A few more focused sessions and you'll be excellent. 📈", "You're building solid foundations. Keep pushing! 🚀", "Consistency is key — you're doing well. Stay focused! ⭐"],
        'needs_improvement'=> ["There's clear room to grow. Focus on your weak subjects now. 💡", "Don't give up — identify the weak areas and attack them daily. 🎯", "Every expert was once a beginner. Fix the gaps systematically! 📚"],
        'critical'         => ["This is a wake-up call. It's not too late — start today! ⏰", "The A/L exam waits for no one. Let's fix this NOW. 💪", "Difficulty is temporary, but your results are permanent. Act now! 🔥"],
    ];
    $pool = $messages[$result['overall_status']];
    $result['motivation'] = $pool[array_rand($pool)];

    return $result;
}

// =============================================
// Get subject trend (improving/declining/stable)
// =============================================
function getSubjectTrend($paperTrend, $subject) {
    $scores = array_column(array_filter($paperTrend, fn($p) => $p['subject'] === $subject), 'pct');
    if (count($scores) < 2) return 'stable';
    $first = $scores[0];
    $last  = end($scores);
    if ($last > $first + 8)  return 'improving';
    if ($last < $first - 8)  return 'declining';
    return 'stable';
}

// =============================================
// CLAUDE API CALL
// =============================================
function callClaudeAPI($data, $rules) {
    $user    = $data['user'];
    $stream  = $user['stream'];
    $avgScore= round($data['avgScore'], 1);
    $weekH   = $data['weekH'];
    $streak  = $data['streak'];

    // Build context summary
    $paperSummary = '';
    foreach ($data['paperPerf'] as $subj => $perf) {
        $paperSummary .= "- $subj: avg {$perf['avg_pct']}% ({$perf['count']} papers, grade {$perf['grade']})\n";
    }
    $hoursSummary = '';
    foreach ($data['subHours'] as $subj => $hrs) {
        $hoursSummary .= "- $subj: {$hrs}h\n";
    }
    $weakList = implode(', ', array_column($rules['weak_subjects'], 'subject'));

    $prompt = <<<PROMPT
You are an expert Sri Lankan A/L (Advanced Level) study coach. Analyze this student's data and give personalized, specific, actionable feedback.

STUDENT PROFILE:
- Name: {$user['name']}
- Stream: $stream
- Exam Year: {$user['exam_year']}
- Target Z-Score: {$user['target_zscore']}

PERFORMANCE DATA:
- Overall average score: $avgScore%
- Study hours this week: {$weekH}h
- Current study streak: {$streak} days
- Study days in last 30 days: {$data['studyDays']}

PAST PAPER RESULTS BY SUBJECT:
$paperSummary

STUDY HOURS BY SUBJECT:
$hoursSummary

WEAK SUBJECTS IDENTIFIED: $weakList
OVERALL STATUS: {$rules['overall_status']}

Please provide:
1. **Personal Assessment** (2-3 sentences specific to this student's situation)
2. **Top 3 Priority Actions** (very specific, numbered, actionable steps for the next 2 weeks)
3. **Subject-Specific Advice** (for each weak subject, give 1 concrete strategy)
4. **Z-Score Outlook** (based on current performance, what's realistic, what needs to change)
5. **One motivational insight** specific to Sri Lankan A/L exam pressure

Be direct, specific, and encouraging. Write in a coaching tone. Keep each section concise. Do NOT use generic advice.
PROMPT;

    $payload = json_encode([
        'model'      => CLAUDE_MODEL,
        'max_tokens' => 1000,
        'messages'   => [['role' => 'user', 'content' => $prompt]],
    ]);

    $ch = curl_init('https://api.anthropic.com/v1/messages');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $payload,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'x-api-key: ' . CLAUDE_API_KEY,
            'anthropic-version: 2023-06-01',
        ],
        CURLOPT_TIMEOUT        => 30,
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode === 200) {
        $decoded = json_decode($response, true);
        return $decoded['content'][0]['text'] ?? '';
    }

    return ''; // Fall back to rules-only if API fails
}

// =============================================
// Cache report in session (avoid re-running)
// =============================================
function getCachedReport($conn, $uid) {
    if (isset($_SESSION['ai_report']) &&
        $_SESSION['ai_report']['uid'] === $uid &&
        (time() - $_SESSION['ai_report']['time']) < 300) { // 5 min cache
        return $_SESSION['ai_report']['data'];
    }
    $report = generateAIReport($conn, $uid);
    $_SESSION['ai_report'] = ['uid' => $uid, 'time' => time(), 'data' => $report];
    return $report;
}

function gradeFromPercent($pct) {
    if ($pct >= 75) return 'A';
    if ($pct >= 65) return 'B';
    if ($pct >= 50) return 'C';
    if ($pct >= 35) return 'S';
    return 'F';
}
?>
